const initializeUpdater = require('@postman/app-updater').init,
      SerializedError = require('serialised-error'),

      electron = require('electron'),

      AppUpdaterAdapter = require('../adapters/AppUpdaterAdapter'),

      APP_UPDATE = 'appUpdate',
      APP_UPDATE_EVENTS = 'app-update-events',
      CHECK_FOR_ELECTRON_VERSION_UPDATED = 'checkForElectronVersionUpdated',
      UPDATE_ELECTRON = 'updateElectron',
      APPLY_ELECTRON_UPDATE = 'applyElectronUpdate',

      ELECTRON_UPDATE_ERROR = 'electronAppUpdateError',
      ELECTRON_UPDATE_NOT_AVAILABLE = 'electronAppUpdateNotAvailable',
      ELECTRON_UPDATE_DOWNLOADED = 'electronAppUpdateDownloaded',
      ELECTRON_VERSION_UPDATED = 'electronVersionUpdated',

      /**
       * @description The initialize method which will initialize the updater module and call the callback based on the updater
       * ===============================================================================================
       * = Callback will not be called if the app start is invalid. Decided by the updater module.init =
       * ===============================================================================================
       */
      init = function (cb) {

        // Initialize updater
        const updaterInstance = initializeUpdater({ adapter: AppUpdaterAdapter });

        // If no updater found for the OS, consider it as not as error, just go through it
        if (!updaterInstance) {
          pm.logger.warn('UpdaterHandler~init - Updater not found for the os');
          cb && cb();
          return;
        }

        updaterInstance.init((err) => {
            attachUpdaterEventsListeners(updaterInstance);
            attachUpdaterListeners(updaterInstance);

            err ? pm.logger.error('UpdateHandler~init - Failed', err) : pm.logger.info('UpdateHandler~init - Success');
            cb && cb(err, updaterInstance);
        });
      },

      /**
       * @description It clears the updaterInstance assigned in the electron.app object
       */
      destroy = function () {
        // Destroys the updaterInstance
        electron.app.updaterInstance = null;
      },

      /**
       * @description It attaches the subscription to the app-update-events for the following events
       * 1. checkForElectronVersionUpdated
       * 2. updateElectron
       * 2. applyElectronUpdate
       */
      attachUpdaterEventsListeners = function (updaterInstance) {
        pm.eventBus.channel(APP_UPDATE_EVENTS).subscribe((event = {}) => {
          pm.logger.info(`UpdateHandler~${APP_UPDATE_EVENTS} - Received event`, event); // Logging intentionally

          let eventName = event.name;
          if (eventName === CHECK_FOR_ELECTRON_VERSION_UPDATED) {
            checkForVersionUpdated(updaterInstance);
            return;
          }
          if (eventName === UPDATE_ELECTRON) {
            downloadUpdate(updaterInstance, event.data);
            return;
          }
        });

        // Sending through bus is not recommended here
        // As, the app quits in this case which triggers a crash at times
        const ipcMain = pm.sdk.IPC;

        ipcMain.subscribe(APPLY_ELECTRON_UPDATE, () => {
          restartAndUpdate(updaterInstance);
        });
      },

      /**
       * @param {Object} updaterInstance
       * It attaches the listeners for the updaterInstance actions.
       */
      attachUpdaterListeners = function (updaterInstance) {
        updaterInstance.on('updateDownloaded', handleOnUpdateDownloaded);
        updaterInstance.on('updateNotAvailable', handleOnUpdateNotAvailable);
        updaterInstance.on('beforeAppQuit', handleAppQuitting);
        updaterInstance.on('versionUpdated', handleOnVersionUpdate);
        updaterInstance.on('error', handleOnUpdateError);
        updaterInstance.on('warning', handleOnUpdateWarning);
      },

      /**
       * =============================================
       * =  Event Handlers for updater events starts =
       * =============================================
       */

      /**
       * @param {Object} event
       * @param {String=} notes
       * @param {String=} releaseName
       * @param {String=} date
       * @param {String=} URL
       */
      handleOnUpdateDownloaded = function (event, notes, releaseName, date, URL) {
        if (releaseName) {
          AppUpdaterAdapter.setDownloadedVersion(releaseName);
        }

        // Note: downloadVersion is not returning a string value from the api.
        // It should have return the downloaded version like x.y.z but it is returning a function
        pm.eventBus.channel(APP_UPDATE_EVENTS).publish({
          name: ELECTRON_UPDATE_DOWNLOADED,
          namespace: APP_UPDATE,
          data: {
            releaseName,
            notes,
            date,
            URL
          }
        });
      },

      handleOnUpdateNotAvailable = function () {
        pm.eventBus.channel(APP_UPDATE_EVENTS).publish({
          name: ELECTRON_UPDATE_NOT_AVAILABLE,
          namespace: APP_UPDATE
        });
      },

      /**
       * @param {Error} error
       * @param {Object=} updateData
       */
      handleOnUpdateError = function (error, updateData) {
        pm.eventBus.channel(APP_UPDATE_EVENTS).publish({
          name: ELECTRON_UPDATE_ERROR,
          namespace: APP_UPDATE,
          data: {
            error: new SerializedError(error),
            updateData
          }
        });
      },

      handleOnUpdateWarning = function (error, info) {
        pm.logger.warn('UpdaterHandler: Warning while updating app', error, info);
      },

      /**
       * @param {?String} lastKnownVersion
       * @param {String} currentVersion
       */
      handleOnVersionUpdate = function (lastKnownVersion, currentVersion) {
        pm.eventBus.channel(APP_UPDATE_EVENTS).publish({
          name: ELECTRON_VERSION_UPDATED,
          namespace: APP_UPDATE,
          data: {
            lastKnownVersion,
            currentVersion
          }
        });
      },

      /**
       * @description This will be called when the updater quits the app.
       */
      handleAppQuitting = function () {
        electron.app.quittingApp = true;
      },

      /**
       * ===========================================
       * =  Event Handlers for updater events ends =
       * ===========================================
       */


      /**
       * ===========================================
       * =  App to updater delegate methods starts =
       * ===========================================
       */

      /**
       * @param {Object} updaterInstance
       * @param {Object} updateInfo
       */
      downloadUpdate = function (updaterInstance, updateInfo) {
        updaterInstance.downloadUpdate(updateInfo);
      },

      /**
       * @param {Object} updaterInstance
       */
      restartAndUpdate = function (updaterInstance) {
        // Marking restart flag as true to relaunch the app after applying updates
        updaterInstance.restartAppToUpdate({ restart: true });
      },

      /**
       * Responsible to apply update and quit the app. This method only supports `Linux` platform
       * @param {Object} updaterInstance
       */
      applyUpdateAndQuit = function (updaterInstance) {

        // @TODO: Fix the wrong implementation of `restartAppToUpdate` in app-updater module.
        // https://postmanlabs.atlassian.net/browse/CFDTN-878

        // Marking restart flag as false to prevent relaunching the app after applying updates
        updaterInstance.restartAppToUpdate({ restart: false });
      },

      /**
       * @param {Object} updaterInstance
       */
      checkForVersionUpdated = function (updaterInstance) {
        updaterInstance.checkForVersionUpdated();
      };

      /**
       * =========================================
       * =  App to updater delegate methods ends =
       * =========================================
       */

module.exports = {
  init,
  applyUpdateAndQuit
 };
