"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connection = void 0;
const events_1 = require("events");
const HTTPAgent_1 = require("./agents/HTTPAgent");
const HTTPSAgent_1 = require("./agents/HTTPSAgent");
const utils_1 = require("./utils");
const socket_io_client_factory_1 = require("./socket-io-client-factory");
class Connection extends events_1.EventEmitter {
    constructor(connectionOptions) {
        super();
        this.reconnectCount = -1;
        this.isConnectionAborted = false;
        this.subscriptions = new Set();
        this.onConnectHandler = this.onConnectHandler.bind(this);
        this.onDisconnectHandler = this.onDisconnectHandler.bind(this);
        this.onErrorHandler = this.onErrorHandler.bind(this);
        // Replace connection options with defaults whereever necessary
        this.connectionOptions = {
            ...connectionOptions,
            reconnectOptions: {
                retryCount: connectionOptions.reconnectOptions?.retryCount ?? 0,
                retryDelay: connectionOptions.reconnectOptions?.retryDelay ?? 0,
            },
        };
        try {
            this.open();
        }
        catch (err) {
            const error = err;
            this.emit('error', { error: { message: error.message } });
            this.emit('end', { aborted: this.isConnectionAborted });
            return;
        }
    }
    createClient() {
        const agent = this.createRequestAgent();
        const url = this.url.href;
        const options = {
            extraHeaders: this.connectionOptions.headers,
            path: this.connectionOptions?.connectOptions?.handshakePath,
            timeout: this.connectionOptions?.connectOptions?.handshakeTimeout,
            agent,
            forceNew: true,
            transports: ['websocket'],
            autoConnect: false,
            reconnection: false,
        };
        return socket_io_client_factory_1.SocketIOClientFactory.getClient(this.connectionOptions.clientVersion)(url, options);
    }
    open() {
        try {
            if (typeof this.connectionOptions.url === 'string') {
                this.url = (0, utils_1.parseUrl)(this.connectionOptions.url);
            }
            else {
                this.url = (0, utils_1.parseUrl)(this.connectionOptions.url.href);
            }
        }
        catch (err) {
            const error = err;
            this.emit('error', { error: { message: error?.message } });
            this.emit('end', { aborted: this.isConnectionAborted });
            return;
        }
        this.client = this.createClient();
        this.client
            .on('connect', this.onConnectHandler)
            .on('disconnect', this.onDisconnectHandler)
            .on('connect_error', this.onErrorHandler);
        this.client.connect();
    }
    createRequestAgent() {
        const { protocol } = this.url;
        const isSecure = protocol === 'https:' || protocol === 'wss:';
        const options = isSecure ?
            {
                rejectUnauthorized: Boolean(this.connectionOptions?.tlsOptions?.rejectUnauthorized),
                ...this.connectionOptions.tlsOptions?.secureContext,
            }
            : {};
        const AgentClass = isSecure ? HTTPSAgent_1.HTTPSAgent : HTTPAgent_1.HTTPAgent;
        const agent = new AgentClass(options);
        agent.on('request-meta', (request) => (this.request = request));
        agent.on('response-meta', (response) => (this.response = response));
        return agent;
    }
    reconnect(reason) {
        this.client.removeAllListeners();
        if (this.reconnectCount >= this.connectionOptions.reconnectOptions.retryCount) {
            this.emit('end', { aborted: this.isConnectionAborted, reason });
            return;
        }
        // Clearing the timeout just to be safe. Most likely any existing timeout would have been executed at this point.
        clearTimeout(this.reconnectTimeoutHandle);
        this.reconnectTimeoutHandle = setTimeout(() => {
            this.open();
        }, this.connectionOptions.reconnectOptions.retryDelay);
        this.emit('reconnect', {
            attempt: ++this.reconnectCount,
            timeout: this.connectionOptions.reconnectOptions.retryDelay,
        });
    }
    messageHandlerFactory(eventName) {
        return (...args) => {
            const messages = args.map((arg) => {
                if (typeof arg === 'string') {
                    return arg;
                }
                if (Buffer.isBuffer(arg)) {
                    const message = arg.buffer.slice(arg.byteOffset, arg.byteOffset + arg.byteLength);
                    return new Uint8Array(message);
                }
                if (ArrayBuffer.isView(arg)) {
                    return new Uint8Array(arg.buffer, arg.byteOffset, arg.byteLength);
                }
                return JSON.stringify(arg);
            });
            this.emit('message', { messages, eventName });
        };
    }
    publish(event, messages, opts) {
        const finalMessages = messages.concat(Boolean(opts?.acknowledgement) ? [this.messageHandlerFactory(event)] : []);
        this.client.emit(event, ...finalMessages);
    }
    subscribe(event) {
        if (this.subscriptions.has(event)) {
            // We don't want to subscribe to the same event multiple times,
            // since that will lead to duplicate messages event to be fired
            // for the same subscriptions due to different handlers being
            // attached to the same event.
            return;
        }
        this.client.on(event, this.messageHandlerFactory(event));
        this.subscriptions.add(event);
        // While this can be a synthetic event, there is no way for the consumer to know if the topics were resubscribed after a reconnect, hence emitting the event from the client
        this.emit('subscribed', { event });
    }
    unsubscribe(event) {
        if (!this.subscriptions.has(event)) {
            return;
        }
        this.client.removeAllListeners(event);
        this.subscriptions.delete(event);
        // While this event can be a synthetic event, this is added for consistency since the client is emitting subscribe events
        this.emit('unsubscribed', { event });
    }
    disconnect() {
        if (!this.client.connected) {
            // This disconnect might be received in the middle of a reconnect attempt, thus clear the timeout to avoid reconnection
            clearTimeout(this.reconnectTimeoutHandle);
            this.isConnectionAborted = true;
            this.emit('end', { aborted: this.isConnectionAborted });
        }
        this.subscriptions.clear();
        this.client.disconnect();
    }
    // region: Client Event Handlers
    // These exist separately to allow for easier testing
    onConnectHandler() {
        this.emit('open', { request: this.request, response: this.response });
        // Subscribe to previous subscriptions if connected after a disconnect
        // We can simply iterate over the topics in subscriptions since it will be empty for the first time connection
        for (const event of this.subscriptions) {
            this.client.on(event, this.messageHandlerFactory(event));
            this.emit('subscribed', { event: event });
        }
        this.reconnectCount = 0;
        this.request = this.response = undefined;
    }
    onDisconnectHandler(reason) {
        if (this.reconnectCount === 0) {
            this.emit('close', { reason });
        }
        // Only try to reconnect if:
        // 1. It was not disconnected manually
        // 2. The connection was not aborted
        // 3. Connected at least once
        if (reason === 'io client disconnect' ||
            this.isConnectionAborted ||
            this.reconnectCount === -1) {
            this.emit('end', { aborted: this.isConnectionAborted, reason });
            return;
        }
        this.reconnect(reason);
    }
    onErrorHandler(error) {
        // If connection was aborted, absorb the error
        if (this.isConnectionAborted) {
            return;
        }
        // Emit error only for the final reconnect attempt
        if (this.reconnectCount > 0 &&
            this.reconnectCount < this.connectionOptions.reconnectOptions.retryCount) {
            return this.reconnect();
        }
        this.emit('error', {
            error: {
                message: (0, utils_1.serializeErrorMessage)(error, this.connectionOptions.clientVersion),
            },
            handshakeRequest: this.request,
            handshakeResponse: this.response,
        });
        this.emit('end', { aborted: this.isConnectionAborted });
    }
}
exports.Connection = Connection;
//# sourceMappingURL=connection.js.map